/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool 
          with fuzzy matching, translation memory, keyword search, 
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2010 Antonio Vilei
               2012 Thomas Cordonnier
               2014 Piotr Kulik, Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.search;

import java.util.Date;
import java.util.List;

import org.omegat.core.data.TMXEntry;
import org.omegat.core.Core;
import org.omegat.util.TMXProp;

/**
 * Storage for a search result entry coming from project TMX (orphans) or other-language entries
 * 
 * @author Antonio Vilei
 * @author Thomas Cordonnier
 */
public class InternalSearchResultEntry extends SearchResultEntry {

    /**
     * Creates a new search result entry with properties set to given values.
     * 
     * @param fileName
     *            Name of TMX file. Can be a dummy name.
     * @param entry
     *            TMX entry
     * @param srcMatch
     *            Matches found in source
     * @param targetMatch
     *            Matches found in target
     * @param noteMatch
     *            Matches found in note
     */
    public InternalSearchResultEntry(String fileName, TMXEntry entry,
            List<SearchMatch> srcMatch, List<SearchMatch> targetMatch, List<SearchMatch> noteMatch) {
        super(srcMatch, targetMatch, noteMatch);
        m_entry = entry;
        m_fileName = fileName;
    }

    /**
     * Returns the number of the corresponding entry within a project. The
     * returned value is < 0 
     */
    public int getEntryNum() {
        return -1;
    }

    /** Returns information about where this entry comes from. */
    public String getPreamble() {
        return "TM:" + getShortPathName() + getMoreString() + ">";
    }
    
    public String getShortPathName() {
        String root = Core.getProject().getProjectProperties().getTMRoot();
        if (m_fileName.startsWith(root)) return m_fileName.substring (root.length());
        return m_fileName;
    }

    /** Returns the source text of the corresponding entry within a project. */
    public String getSrcText() {
        return m_entry.source;
    }

    /** Returns the target text of the corresponding entry within a project. */
    public String getTranslation() {
        return m_entry.translation;
    }

    /** Returns the note text of the corresponding entry within a project. */
    public String getNote() {
        return m_entry.note;
    }

    public String getFileName() {
        return m_fileName;
    }
    
    public String getAuthor() {
        return m_entry.creator;
    }

    public Date getDate() {
        return new Date(Math.max(m_entry.creationDate, m_entry.changeDate));
    }

    public List<TMXProp> getProperties() {
        return java.util.Collections.emptyList();
    }

    public String getRevisor() {
        return m_entry.revisor;
    }

    private String m_fileName;
    private TMXEntry m_entry;
}
