/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool 
          with fuzzy matching, translation memory, keyword search, 
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2010 Antonio Vilei
               2012 Thomas Cordonnier
               2014 Piotr Kulik, Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.search;

import java.util.Date;
import java.util.List;

import org.omegat.util.TMXProp;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.data.TMXEntry;

/**
 * Storage for a search result entry coming from project memory, not orphan
 *
 * @author Antonio Vilei
 * @author Thomas Cordonnier
 */
public class ReplaceSearchResultEntry extends SearchResultEntry {

    /**
     * Creates a new search result entry with properties set to given values.
     * 
     * @param sourceEntry
     *            The source entry this result comes from. Contains source and comment
     * @param traEntry
     *            Associated translation. May be null
     */
    public ReplaceSearchResultEntry(SourceTextEntry sourceEntry, TMXEntry traEntry, List<SearchMatch> targetMatch) {
        m_targetMatch = targetMatch;
        m_tmxEntry = traEntry;
        m_sourceEntry = sourceEntry;
    }

    /**
     * Returns the number of the corresponding entry within a project. The
     * returned value is > 0, as the entry belongs to one of the source files of
     * the project; 
     */
    public int getEntryNum() {
        return m_sourceEntry.entryNum();
    }

    /** Returns information about where this entry comes from. */
    public String getPreamble() {
        return "" + getEntryNum() + getMoreString();
    }
    
    /** Returns the source text of the corresponding entry within a project. */
    public String getSrcText() {
        return  m_sourceEntry.getSrcText();
    }

    /** Returns the target text of the corresponding entry within a project. */
    public String getTranslation() {
        return m_tmxEntry.translation == null ?  "" : m_tmxEntry.translation;
    }

    /** Returns the note text of the corresponding entry within a project. */
    public String getNote() {
        return m_tmxEntry.note == null ?  "" : m_tmxEntry.note;
    }

    public String getFileName() {
        return m_sourceEntry.getKey().file;
    }
    
    public String getAuthor() {
        return m_tmxEntry.creator == null ?  "" : m_tmxEntry.creator;
    }

    public Date getDate() {
        return new Date (m_tmxEntry.creationDate);
    }

    public List<TMXProp> getProperties() {
        return java.util.Collections.emptyList();
    }
	
    public String getRevisor() {
        return m_tmxEntry.revisor;
    }		
    
    public List<SearchMatch> getSrcMatch() {
        return null;
    }

    public List<SearchMatch> getTargetMatch() {
        return m_targetMatch;
    }

    public List<SearchMatch> getNoteMatch() {
        return null;
    }
    
    /** In replacement, duplicate is only for default entries **/
    @Override
    public boolean equals (SearchResultEntry entry) {
        if (! super.equals(entry)) return false;
        try {
            ReplaceSearchResultEntry rEntry = (ReplaceSearchResultEntry) entry;
            if (m_tmxEntry.defaultTranslation) return rEntry.m_tmxEntry.defaultTranslation;
            return false; // do not merge alternative translations, they can differ
        } catch (ClassCastException cce) {
            return false;
        }
    }
    
	    
    private List<SearchMatch> m_targetMatch;		
    private SourceTextEntry m_sourceEntry;
    private TMXEntry m_tmxEntry;
}
