/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool
          with fuzzy matching, translation memory, keyword search,
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2012-2016 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.filehooks;

import org.omegat.core.data.SourceTextEntry;

/**
 * Enable to associate events before display of source or target document
 * 
 * @author Thomas CORDONNIER
 */
public interface IFileHook {
	
	// ------------------ Interface part ---------------------
	
	/**
	 * Returns which file should be opened by 'view source' menu <br/>
	 * Usually the source file itself, except if it is a container (such as xliff) <br/>
	 * Note: this method is <i>not</i> an accessor, it may take time, for example to extract the file to be read
	 **/
	public default String getNativeSourceFile (String currentFile, SourceTextEntry entry) { return currentFile; }
	
	/** Indicates whenever current hook does something with pseudo-tags: if not, menus should not be generated **/
	public default boolean supportsPseudoTags() { return false; }
	
	/** Method called after compilation of a single file **/
	public default void postCompile(java.io.File destDir, String midName) {}
	
	/** Method called before compilation of a single file: if true, this file is not compiled **/
	public default boolean isNotToCompile (String midName) { return false; }

	/**
	 * Returns which file should be opened by 'view target' menu <br/>
	 * Usually the target file itself, except if it is a container (such as xliff) <br/>
	 * Note: this method must be called <i>after</i> postCompile, else it may return null
	 **/
	public default String getNativeTargetFile (String currentFile) { return currentFile; }
	
	/** Indicates whenever files using this hook can be opened in Trados Studio **/
	public default boolean supportsOpenInStudio(String currentFile) { return false; }

	/**
	 * Returns the file to be opened in Trados Studio. Only called if supportsOpenInStudio = true
	 **/
	public default String getTradosTargetFile (String currentFile) { return currentFile; }
	
	/** After call to postCompile, indicates whenever some exception occured **/
	public default Exception lastCompileException() { return null; }
	
	// ------------------ Factory -------------------
	
	public static final IFileHook DEFAULT_HOOK = new IFileHook() {};
	public static final XliffFileHook XLIFF_HOOK = new XliffFileHook();
	public static final SdlxliffFileHook SDLXLIFF_HOOK = new SdlxliffFileHook();
	public static final ReformatterHook REFORMAT_HOOK = new ReformatterHook();
	public static final RenumXliffHook RENUM_XLIFF_HOOK = new RenumXliffHook();
	public static final SdlProjectHook SDLPROJ_HOOK = new SdlProjectHook();
	
	public static IFileHook hookForFile (String currentFile) {
		if (currentFile.endsWith(".sdlxliff")) 
			return SDLXLIFF_HOOK;
		if (currentFile.endsWith (".renum.xlf")) 
			return RENUM_XLIFF_HOOK;
		if (currentFile.endsWith (".xlf") || currentFile.endsWith(".xliff")) 
			return XLIFF_HOOK;
		if (currentFile.endsWith (".docx") || currentFile.endsWith(".htm") || currentFile.endsWith(".html")) 
			return REFORMAT_HOOK;
		if (currentFile.endsWith (".sdlppx")) 
			return SDLPROJ_HOOK;
		return DEFAULT_HOOK;
	}
	
}
