/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool 
          with fuzzy matching, translation memory, keyword search, 
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2015 Thomas Cordonnier
               2017 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.gui.editor.filter;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.omegat.core.Core;
import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.data.TMXEntry;
import org.omegat.core.search.SearchMatch;
import org.omegat.core.search.SearchResultEntry;
import org.omegat.core.search.PreTranslateSearcher;
import org.omegat.core.search.PreTranslateSearchResultEntry;
import org.omegat.gui.editor.EditorController;
import org.omegat.gui.editor.IEditorFilter;

/**
 * Editor filter implementation.
 * 
 * @author Alex Buloichik (alex73mail@gmail.com)
 * @author Thomas Cordonnier
 */
public class PreTranslateFilter implements IEditorFilter {
    private final Map<Integer, PreTranslateSearchResultEntry> entries = new HashMap<>();
    private FilterBarReplace controlComponent;
    private PreTranslateSearcher searcher;
    private int minEntryNum, maxEntryNum;

    public PreTranslateFilter(PreTranslateSearcher searcher) {
        this.searcher = searcher;

        minEntryNum = Integer.MAX_VALUE;
        maxEntryNum = Integer.MIN_VALUE;
		for (SearchResultEntry sre: searcher.getSearchResults()) {
            minEntryNum = Math.min(minEntryNum, sre.getEntryNum());
            maxEntryNum = Math.max(maxEntryNum, sre.getEntryNum());
			entries.put(sre.getEntryNum(), (PreTranslateSearchResultEntry) sre);
		}		
        controlComponent = new FilterBarReplace();
        

        controlComponent.btnCancel.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                Core.getEditor().commitAndDeactivate(); // Make sure that any change done in the current segment is not lost
                Core.getEditor().removeFilter();
            }
        });
        controlComponent.btnSkip.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                skip();
            }
        });
        controlComponent.btnReplaceNext.setText("Translate next");
        controlComponent.btnReplaceNext.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                translate();
            }
        });
    }

    @Override
    public boolean isSourceAsEmptyTranslation() {
        return true;
    }

    /**
     * PreTranslate all occurrences in all entries.
     */
    public void translateAll(javax.swing.JLabel label) {
        EditorController ec = (EditorController) Core.getEditor(); ec.commitAndDeactivate(); // Make sure that any change done in the current segment is not lost
        int i = 0; boolean transDefault = ! searcher.translateAsAlternative();
		for (PreTranslateSearchResultEntry sre : entries.values()) {
            SourceTextEntry ste = Core.getProject().getAllEntries().get (sre.getEntryNum() - 1);
            if (transDefault && ste.getDuplicate() == SourceTextEntry.DUPLICATE.NEXT) continue; // already replaced
            
            // Still here ? go...
            PrepareTMXEntry prepare = new PrepareTMXEntry(Core.getProject().getTranslationInfo(ste));
            prepare.translation = sre.getTranslationResult();
            Core.getProject().setTranslation(ste, prepare, transDefault, null);
			
			if (label != null) label.setText ("" + (++i / entries.size()) + "/" + entries.size() + " translated."); 
        }
        ec.refreshEntries(entries.keySet()); ec.activateEntry();
    }

    @Override
    public boolean allowed(SourceTextEntry ste) {
        return entries.containsKey(ste.entryNum());
    }

    @Override
    public Component getControlComponent() {
        return controlComponent;
    }

    public List<SearchMatch> getMatchesForEntry(String srcText) {
        return searcher.searchInSource(srcText);
    }

    private void skip() {
        EditorController ec = (EditorController) Core.getEditor();

        // try to find in current entry
        /*int pos = ec.getCurrentPositionInEntryTranslation();
        String str = ec.getCurrentTranslation();
        SearchMatch[]  found = getMatchesForEntry(str);
        if (found != null) {
            for (SearchMatch m : found) {
                if (m.getStart() > pos) {
                    ec.setCaretPosition(new EditorController.CaretPosition(m.getStart(), m.getEnd()));
                    ec.requestFocus();
                    return;
                }
            }
        }*/
        // not found in current entry - find next entry
        int currentEntryNumber = ec.getCurrentEntryNumber();
        List<SearchMatch> found;
        ec.commitAndDeactivate();

        // find to the end of project
        for (int i = currentEntryNumber + 1; i <= maxEntryNum; i++) {
            PreTranslateSearchResultEntry sre = entries.get(i);
            if (sre == null) continue; // entry not filtered
            TMXEntry en = Core.getProject().getTranslationInfo(Core.getProject().getAllEntries().get(i));
            String trans = sre.getSrcText();
            if (trans == null) continue;
            found = getMatchesForEntry(trans);
            if (found == null) continue; // no replacements
            for (SearchMatch m : found) {
                ec.gotoEntry(i, new EditorController.CaretPosition(m.getStart(), m.getEnd()));
                ec.requestFocus();
                return;
            }
        }
        // find from the beginning of project
        for (int i = minEntryNum; i < currentEntryNumber; i++) {
            PreTranslateSearchResultEntry sre = entries.get(i);
            if (sre == null) continue; // entry not filtered
            TMXEntry en = Core.getProject().getTranslationInfo(Core.getProject().getAllEntries().get(i));
            String trans = sre.getSrcText();
            if (trans == null) continue;
            found = getMatchesForEntry(trans);
            if (found == null) continue; // no replacements
            for (SearchMatch m : found) {
                ec.gotoEntry(i, new EditorController.CaretPosition(m.getStart(), m.getEnd()));
                ec.requestFocus();
                return;
            }
        }
        // not found
        ec.activateEntry();
    }

    private void translate() {
        EditorController ec = (EditorController) Core.getEditor();

        // is caret inside match ?
        int pos = ec.getCurrentPositionInEntryTranslation();
        String str = ec.getCurrentTranslation();
        
        PreTranslateSearchResultEntry rEntry = searcher.testSingleEntry(ec.getCurrentEntry());
        if (rEntry != null) {
			// replace full text. If user selected something partial, this has been calculated in searcher
			ec.replaceEditText(rEntry.getTranslationResult());
			ec.setAlternateTranslationForCurrentEntry(searcher.translateAsAlternative());
        }
        
        // skip to next
        skip();
    }

}
