/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool
          with fuzzy matching, translation memory, keyword search,
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2012 Martin Fleurke
               2013 Alex Buloichik (alex73mail@gmail.com)
               2017 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.gui.editor.mark;

import java.util.ArrayList;
import java.util.List;

import javax.swing.event.DocumentEvent;
import javax.swing.text.AttributeSet;
import javax.swing.text.Highlighter.HighlightPainter;

import org.omegat.gui.editor.Document3;
import org.omegat.gui.editor.MarkerController;
import org.omegat.core.Core;
import org.omegat.core.data.SourceTextEntry;

/**
 * Abstract marker class that marks source and target text according to char
 * and paints them accoriding to some 'PAINTER' and 'ATTRIBUTES' (all
 * of they defined by implementing classes)
 * This is easy to implement as a fast marker because only characters added or removed by the event are affected
 * 
 * @author Martin Fleurke
 * @author Alex Buloichik (alex73mail@gmail.com)
 * @author Thomas Cordonnier
 */
public abstract class CharMarker implements IFastMarker {
    protected HighlightPainter PAINTER;
    protected String toolTip;
    protected AttributeSet ATTRIBUTES;
    /** char for mark. */
    protected char patternChar;

    public CharMarker() throws Exception {
    }

    /**
     * Is the marker enabled?
     * 
     * @return true when enabled (markers are painted). false when disabled (no
     *         markers painted)
     */
    protected abstract boolean isEnabled();

    @Override
    public List<Mark> getMarksForEntry(SourceTextEntry ste, String sourceText, String translationText, boolean isActive)
            throws Exception {

        if (!isEnabled()) {
            return null;
        }
        if (patternChar == 0) {
            return null;
        }

        List<Mark> r = new ArrayList<Mark>();
        if (isActive || Core.getEditor().getSettings().isDisplaySegmentSources() || translationText == null) {// TODO
            initDrawers(true, isActive);
            if (patternChar != 0) {
                int pos = -1;
                while ((pos = sourceText.indexOf(patternChar, pos + 1)) >= 0) {
                    Mark m = new Mark(Mark.ENTRY_PART.SOURCE, pos, pos + 1);
                    m.painter = PAINTER;
                    m.toolTipText = toolTip;
                    m.attributes = ATTRIBUTES;
                    r.add(m);
                }
            }
        }
        if (translationText != null) {
            initDrawers(false, isActive);
            if (patternChar != 0) {
                int pos = -1;
                while ((pos = translationText.indexOf(patternChar, pos + 1)) >= 0) {
                    Mark m = new Mark(Mark.ENTRY_PART.TRANSLATION, pos, pos + 1);
                    m.painter = PAINTER;
                    m.toolTipText = toolTip;
                    m.attributes = ATTRIBUTES;
                    r.add(m);
                }
            }
        }

        return r;
    }	

    /**
     * Marker can override this method for define different PAINTER, ATTRIBUTES
     * and PATTERN, based on source/translation, active/inactive and other.
     */
    protected void initDrawers(boolean isSource, boolean isActive) {
    }
    
    public List<MarkerController.MarkInfo> toRemove (DocumentEvent ev, List<MarkerController.MarkInfo> oriList) {
        if (!isEnabled()) return java.util.Collections.EMPTY_LIST;
        if (ev.getType() == DocumentEvent.EventType.INSERT) return java.util.Collections.EMPTY_LIST;
        initDrawers(false, true); // target, active
        
        List<MarkerController.MarkInfo> res = new ArrayList<MarkerController.MarkInfo> (ev.getLength() / 2 + 1);
        for (MarkerController.MarkInfo ori: oriList)
            if (ori.highlight != null)
                if ( ori.highlight.getStartOffset() == ori.highlight.getEndOffset() )
                    res.add (ori);
        return res;
    }
    
    public List<Mark> toAdd (DocumentEvent ev) throws Exception {
        if (!isEnabled()) return java.util.Collections.EMPTY_LIST;
        if (ev.getType() == DocumentEvent.EventType.REMOVE) return java.util.Collections.EMPTY_LIST;
        initDrawers(false, true); // target, active

        int pos = -1; int shift = ev.getOffset() - ((Document3) ev.getDocument()).getTranslationStart();
        if (shift < 0) return java.util.Collections.EMPTY_LIST; 
        if (shift > ((Document3) ev.getDocument()).getTranslationEnd()) return java.util.Collections.EMPTY_LIST;
        List<Mark> res = new ArrayList<Mark> (ev.getLength() / 2 + 1);
        String addedContents = ev.getDocument().getText(ev.getOffset(), ev.getLength());
        while ((pos = addedContents.indexOf(patternChar, pos + 1)) >= 0) {
            Mark m = new Mark(Mark.ENTRY_PART.TRANSLATION, pos + shift, pos + shift + 1);
            m.painter = PAINTER;
            m.toolTipText = toolTip;
            m.attributes = ATTRIBUTES;
            res.add(m);
        }
        return res;
    }	
}
