/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool
          with fuzzy matching, translation memory, keyword search,
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2020 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.glossaries;

import java.util.List;

import org.omegat.util.TMXProp;

/**
 * Generic storage of glossary entry with properties
 * (some formats may have specific, optimized storage method) <br/>
 *
 * Properties are ordered as in the file. We make distinction between properties associated to 
 * the entry itself, and those only associated to source or target term.
 * 
 * @author Thomas Cordonnier
 */
public class GlossaryEntryWithProperties extends GlossaryEntryStore {
    public GlossaryEntryWithProperties(String src, String loc, String note, 
        List<TMXProp> entryProp, List<TMXProp> srcProp, List<TMXProp> traProp,
        IGlossary origin) {
        super(src, loc, note, origin);
        this.srcProp = srcProp; this.traProp = traProp; this.entryProp = entryProp;
    }

    /** 
     * Return the comment string: 
     * contrarily to getNoteText(), this contains properties.
     */
    public String getCommentText() {        
        StringBuffer buf = new StringBuffer();
        for (TMXProp pe: entryProp) buf.append(pe.getType()).append(": ").append(pe.getValue()).append("\n");
        for (TMXProp pe: srcProp) buf.append(pe.getType()).append(": ").append(pe.getValue()).append("\n");
        for (TMXProp pe: traProp) buf.append(pe.getType()).append(": ").append(pe.getValue()).append("\n");
        return buf.toString();
    }
    
    /**
     * For layout scripts: property key can be prefixed by src:, tra:, entry: or *:
     * @return property value if found, null instead
     **/
    public String findProperty(String key) {
        if (! key.contains(":")) key = "*:" + key;
        String prefix = key.substring(0, key.indexOf(':')).toLowerCase(), suffix = key.substring(key.indexOf(':') + 1); 
        if (key.startsWith("e") || key.startsWith("*")) { String val = findProperty(entryProp, suffix); if (val != null) return val; }
        if (key.startsWith("t") || key.startsWith("*")) { String val = findProperty(traProp, suffix); if (val != null) return val; }
        if (key.startsWith("s") || key.startsWith("*")) { String val = findProperty(srcProp, suffix); if (val != null) return val; }
        return null;
    }
    
    private static String findProperty(List<TMXProp> theList, String key) {
        for (TMXProp pe: theList) if (pe.getType().equals(key)) return pe.getValue();
        return null;
    }
    
    public Iterable<TMXProp> getAllProperties() {
        return () -> java.util.stream.Stream.concat(entryProp.stream(), java.util.stream.Stream.concat(srcProp.stream(), traProp.stream())).iterator();
    }
    

    private final List<TMXProp> entryProp, srcProp, traProp;
}
