/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool 
          with fuzzy matching, translation memory, keyword search, 
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2012 Thomas Cordonnier
               2013 Aaron Madlon-Kay, Alex Buloichik
               2014 Alex Buloichik, Piotr Kulik, Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.search;

import java.util.List;
import java.util.Iterator;
import org.omegat.core.data.IProject;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.data.TMXEntry;
import org.omegat.gui.search.ReplaceDialog;


/**
 * Specific searcher for searches in a project
 * 
 * @author Alex Buloichik (alex73mail@gmail.com)
 * @author Aaron Madlon-Kay
 * @author Piotr Kulik
 * @author Thomas Cordonnier
 */
public class ReplaceSearcher extends ProjectSearcher {

    /**
     * Create new searcher instance.
     * 
     * @param project
     *            Current project
     */
    public ReplaceSearcher(final ReplaceDialog window, final IProject project, 
        int numberOfResults, boolean withOrphans,
        TextExpression searchTarget, 
        TextExpression author, TextExpression translator, long dateAfter, long dateBefore) {

        super (window, project, true, false, numberOfResults, 
            author, null, dateAfter, dateBefore);
		
        this.m_searchTarget = searchTarget; 
        this.m_orphans = withOrphans;
    }

	@Override
    protected void testOngoing(SourceTextEntry ste) {
        TMXEntry tmxEntry = m_project.getTranslationInfo(ste);
        if (! getTranslationStateFilter().isValidEntry (tmxEntry)) return;
        if (! checkFilters (tmxEntry)) return;

        List<SearchMatch> targetMatches = m_searchTarget.searchString(tmxEntry.translation);
        if (targetMatches != null) addEntry (new ReplaceSearchResultEntry(ste, tmxEntry, targetMatches));
    }
        
    // Used for replacements
    @Override public TranslationStateFilter getTranslationStateFilter() { return TranslationStateFilter.TRANSLATED_ONLY;  }
    @Override public boolean searchOn(int location) { 
        if ((location & SEARCH_SCOPE_ORPHANS) > 0) return m_orphans;
        return (location & SEARCH_SCOPE_ONGOING) > 0; 
    }
	
    // Used for replacements
    public List<SearchMatch> searchInTranslation(String text) {
        return m_searchTarget == null ? null : m_searchTarget.searchString(text);
    }
    
    // if iterator exists, use it to avoid parsing again begin of the list
    @Override public Iterator<SourceTextEntry> projectEntries() { 
        if (iter == null) iter = m_project.getAllEntries().iterator(); 
        return iter;
    }
    
    public void startAt (Iterator<SourceTextEntry> iter) { this.iter = iter; }
    
    private Iterator<SourceTextEntry> iter = null;   	
    protected final TextExpression m_searchTarget;
    private boolean m_orphans;
}
