/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool
          with fuzzy matching, translation memory, keyword search,
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2000-2006 Keith Godfrey and Maxym Mykhalchuk
               2013 Aaron Madlon-Kay, Alex Buloichik
               2014-2021 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.glossaries;

import java.util.Collections;
import java.util.Collection;

import org.omegat.core.Core;
import org.omegat.core.data.ITranslationEntry;
import org.omegat.gui.glossary.GlossaryEntryView;
import org.omegat.gui.glossary.MergedGlossaryEntries;
import org.omegat.util.StringUtil;

/**
 * An entry as stored in a file or distant glossary: 
 * glossaries cannot decide for merging policy, so this cannot be a merged entry.
 * That said, this class is not final, so more complex glossaries (as TBX) can contain properties.
 * 
 * @author Keith Godfrey
 * @author Aaron Madlon-Kay
 * @author Alex Buloichik
 * @author Thomas Cordonnier
 */
public class GlossaryEntryStore extends GlossaryEntryView implements ITranslationEntry {
    public GlossaryEntryStore(String src, String loc, String note, IGlossary origin) {
        super(src);
        mTarget = StringUtil.normalizeUnicode(loc);
        mNote = StringUtil.normalizeUnicode(note);
        mOrigin = origin;
    }

    /** Return the target-language term string. */
    public final String getLocText() {
        return mTarget;
    }
    
    /** 
     * Return the comment string.
     * Identical to note except in subclasses,
     * where it could be calculated based on various properties.
     */
    public String getCommentText() {        
        return mNote;
    }
    
    /** 
     * Return the note stored for this entry. <br/>
     * Note means a simple text. This may differ from comment, 
     * which in subclasses may be calculated based on various properties.
     */
    public final String getNoteText() {
        return mNote;
    }
    
    public final boolean getPriority() {
        return (mOrigin == Core.getGlossaryManager().getPriorityGlossary());
    }
    
    public final IGlossary getOrigin() {
        return mOrigin;
    }
    
    // ------------------ Compatibility with GlossaryEntryView ----------------------
    
    public final Collection<GlossaryEntryStore> getMergedEntries() {
        return Collections.singletonList(this);
    }
    
    public final Collection<String> getLocTerms(boolean unique) {
        return Collections.singletonList(mTarget);		
    }
    
    public final Iterable<GlossaryEntryStore> getEntriesFor(String tra) {
        if (tra.equals(this.mTarget)) return Collections.singletonList(this); else return Collections.emptyList();
    }
    
    public final boolean hasPriorities() {
        return (mOrigin == Core.getGlossaryManager().getPriorityGlossary());
    }
    
    public final boolean hasPriorities(String tra) {
        return tra.equals(mTarget) && hasPriorities();
    }
    
    // ------------------ implement ITranslationEntry ----------------------
    
    public final String getSourceText() {
        return getSrcText();
    }
    
    public final String getTranslationText() {
        return mTarget;
    }
    
    // ------------------ Hashtable -------------------

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        try {
            GlossaryEntryStore otherStore = (GlossaryEntryStore) o;
            return StringUtil.equalsWithNulls(this.getSrcText(), otherStore.getSrcText())
                    && StringUtil.equalsWithNulls(this.mTarget, otherStore.mTarget)
                    && StringUtil.equalsWithNulls(this.mNote, otherStore.mNote);
        } catch (ClassCastException cce1) {
            try {
                MergedGlossaryEntries otherView = (MergedGlossaryEntries) o;
                Collection<GlossaryEntryStore> subEntries = otherView.getMergedEntries();
                return (subEntries.size() == 1) && (this.equals(subEntries.iterator().next()));
            } catch (ClassCastException cce2) {
                return false;
            }
        }
    }
    
    @Override
    public int hashCode() {
        int hash = 98;
        hash = hash * 17 + (getSrcText() == null ? 0 : getSrcText().hashCode());
        hash = hash * 31 + (mTarget == null ? 0 : mTarget.hashCode());
        hash = hash * 13 + (mNote == null ? 0 : mNote.hashCode());
        return hash;
    }

    private final String mTarget;
    private final String mNote;
    private final IGlossary mOrigin;
}
