/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool 
          with fuzzy matching, translation memory, keyword search, 
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2008 Alex Buloichik
               2021 Thomas Cordonnier
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.filters2.subtitles;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import org.omegat.core.data.ITMXEntry;
import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.filters2.AbstractFilter;
import org.omegat.filters2.FilterContext;
import org.omegat.filters2.Instance;
import org.omegat.filters2.TranslationException;
import org.omegat.util.MixedEolHandlingReader;
import org.omegat.util.NullBufferedWriter;
import org.omegat.util.OStrings;
import org.omegat.util.StringUtil;

/**
 * Filter for subtitles files. Format described on
 * http://en.wikipedia.org/wiki/SubRip.
 * 
 * @author Alex Buloichik (alex73mail@gmail.com)
 * @author Thomas Cordonnier
 */
public class SrtFilter extends AbstractFilter {
    protected static final Pattern PATTERN_TIME_INTERVAL = Pattern
            .compile("([0-9]{2}:[0-9]{2}:[0-9]{2},[0-9]{3})\\s+-->\\s+([0-9]{2}:[0-9]{2}:[0-9]{2},[0-9]{3})");
    protected static final Pattern PATTERN_ID = Pattern .compile("^\\d+$");
    protected static final String EOL = "\r\n";

    enum READ_STATE {
        WAIT_ID, WAIT_TIME, WAIT_TEXT
    };

    protected Map<String, String> align;

    protected String key, start, end;
    protected StringBuilder text = new StringBuilder();
    protected BufferedWriter out;

    @Override
    public Instance[] getDefaultInstances() {
        return new Instance[] { new Instance("*.srt") };
    }

    @Override
    public String getFileFormatName() {
        return OStrings.getString("SRTFILTER_FILTER_NAME");
    }

    @Override
    public boolean isSourceEncodingVariable() {
        return true;
    }

    @Override
    public boolean isTargetEncodingVariable() {
        return true;
    }

    @Override
    protected void processFile(BufferedReader inFile, BufferedWriter outFile, FilterContext fc) throws IOException,
            TranslationException {
        out = outFile;
        READ_STATE state = READ_STATE.WAIT_ID;
        key = null;
        text.setLength(0);

        MixedEolHandlingReader reader = new MixedEolHandlingReader(inFile);

        String s; Matcher m;
        while ((s = reader.readLine()) != null) {
            String trimmed = s.trim();
            switch (state) {
            case WAIT_ID:
                if (PATTERN_ID.matcher(trimmed).matches()) {
                    key = trimmed; state = READ_STATE.WAIT_TIME;
                }
                text.setLength(0);
                break;
            case WAIT_TIME: {
                m = PATTERN_TIME_INTERVAL.matcher(trimmed);
                if (m.matches()) {
                    state = READ_STATE.WAIT_TEXT; start = m.group(1); end = m.group(2);
                }
                text.setLength(0);
                break;
            }
            case WAIT_TEXT:
                if (trimmed.isEmpty()) {
                    flush();
                    outFile.write(EOL);
                    state = READ_STATE.WAIT_ID;
                }
                if (text.length() > 0) {
                    text.append('\n');
                }
                text.append(s);
                break;
            }
        }

        reader.close();
        flush();
    }

    private void flush() throws IOException {
        if (text.length() == 0) {
            return;
        }

        if (align != null) {
            align.put(key, text.toString());
        }
        
        if (entryParseCallback != null) {
            entryParseCallback.addEntry(key, text.toString(), null, false, start + " --> " + end, null, this, null);
        } else {
            ITMXEntry itr = entryTranslateCallback.getTranslationEntry(key, text.toString(), null);
            if (itr == null) {
               out.write(start + " --> " + end + EOL);
               out.write(text.toString().replace("\n", EOL));
            } else {
               if (itr.getNote() != null) {
                   Matcher m = PATTERN_TIME_INTERVAL.matcher(itr.getNote());
                   if (m.find()) { start = m.group(1); end = m.group(2); }
               }
               out.write(start + " --> " + end + EOL);
               String tra = itr.getTranslationText(); if (tra == null) tra = text.toString();
               out.write(tra.replace("\n", EOL));
            }
        }
        
        key = null;
        text.setLength(0);
    }

    @Override
    protected void alignFile(BufferedReader sourceFile, BufferedReader translatedFile, FilterContext fc) throws Exception {
        Map<String, String> source = new HashMap<String, String>();
        Map<String, String> translated = new HashMap<String, String>();

        align = source;
        processFile(sourceFile, new NullBufferedWriter(), fc);
        align = translated;
        processFile(translatedFile, new NullBufferedWriter(), fc);
        for (Map.Entry<String, String> en : source.entrySet()) {
            String tr = translated.get(en.getKey());
            if (!StringUtil.isEmpty(tr)) {
                entryAlignCallback.addTranslation(en.getKey(), en.getValue(), tr, false, null, this);
            }
        }
    }
}
