/**************************************************************************
 OmegaT - Computer Assisted Translation (CAT) tool
          with fuzzy matching, translation memory, keyword search,
          glossaries, and translation leveraging into updated projects.

 Copyright (C) 2017 Thomas CORDONNIER
               Home page: http://www.omegat.org/
               Support center: http://groups.yahoo.com/group/OmegaT/

 This file is part of OmegaT.

 OmegaT is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 OmegaT is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.core.matching.external;

import org.omegat.core.data.ITMXEntry;
import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.Core;
import org.omegat.core.CoreEvents;
import org.omegat.core.events.IProjectEventListener;
import org.omegat.util.Language;

import java.util.Map;
import java.util.Iterator;

/**
 * Adds capacity to memorize non-saved entries, to send them later
 * 
 * @author Thomas CORDONNIER
 */
public class CacheWritableExternalMemory implements IWritableExternalMemory {

	private IWritableExternalMemory delegate;
	private Map<SourceTextEntry,PrepareTMXEntry> failedEntries = new java.util.HashMap<>();

	public CacheWritableExternalMemory(IWritableExternalMemory delegate) {
		this.delegate = delegate;
		CoreEvents.registerProjectChangeListener(eventType -> {
			if (eventType == IProjectEventListener.PROJECT_CHANGE_TYPE.CLOSE || eventType == IProjectEventListener.PROJECT_CHANGE_TYPE.SAVE)
				try {
					Language sLang = Core.getProject().getProjectProperties().getSourceLanguage();
					Language tLang = Core.getProject().getProjectProperties().getTargetLanguage();
					delegate.registerTranslations(sLang, tLang, failedEntries);
					failedEntries.clear(); // only if there was no exception
				} catch (Exception e) {
					
				}
		});
	}

	// Implement methods from IWritableExternalMemory as delegation
	
	public String getProviderName() { return delegate.getProviderName(); }

	public String getMemoryName() { return delegate.getMemoryName(); }
		
    public boolean isWriteMode() { return delegate.isWriteMode(); }
    
    public boolean mustWrite(PrepareTMXEntry entryContents, SourceTextEntry entrySource) { 
		return delegate.mustWrite(entryContents,entrySource);
	}
    
    public void registerTranslations(Language sLang, Language tLang, Map<SourceTextEntry,PrepareTMXEntry> entries) throws Exception {
		delegate.registerTranslations(sLang, tLang, entries);
	}

	// Now the methods which differ from delegation
	
    /** Also tries to send failed translations */
    public void registerTranslation(Language sLang, Language tLang, PrepareTMXEntry entryContents, SourceTextEntry entrySource) throws Exception {
		try {
			delegate.registerTranslations(sLang, tLang, failedEntries);
			failedEntries.clear(); // only if there was no exception
		} finally {
			try { 
				delegate.registerTranslation(sLang, tLang, entryContents, entrySource);
			} catch (Exception e) {
				failedEntries.put (entrySource, entryContents);
			}
		}
	}
    
	/** Also return non-saved values, which are susceptible to match as well **/
	public Iterable<ITMXEntry> findMatchingTranslations (Language sLang, Language tLang, String text, int minScore, int maxCount) throws Exception {
		final Iterator<? extends ITMXEntry> delegateIterator = delegate.findMatchingTranslations(sLang, tLang, text, minScore, maxCount).iterator();
		return () -> new Iterator<ITMXEntry>() {
			Iterator<PrepareTMXEntry> cacheIterator = failedEntries.values().iterator();
			
			public boolean hasNext() { return cacheIterator.hasNext() || delegateIterator.hasNext(); }
			
			public void remove() { }
			
			public ITMXEntry next() {
				if (cacheIterator.hasNext()) return cacheIterator.next(); else return delegateIterator.next();
			}
		};
	}
}
